package com.androidbook.simplebluetooth;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Set;
import java.util.UUID;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothServerSocket;
import android.bluetooth.BluetoothSocket;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.media.MediaPlayer;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.util.Log;
import android.view.View;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.ListView;
import android.widget.TextView;
import android.widget.ToggleButton;
import android.widget.AdapterView.OnItemClickListener;

public class SimpleBluetooth extends Activity {
    private static final String SIMPLE_BT_RESPONSE_POLO = "polo";
    private static final String SIMPLE_BT_COMMAND_MARCO = "marco";
    private static final String SIMPLE_BT_COMMAND_PING = "ping";
    private static final int BT_DISCOVERABLE_DURATION = 300; // maksymalny czas, użyteczne do debugowania //120;
    private static final String DEBUG_TAG = "SimpleBluetooth";
    // generowany losowo; poszukaj w wyszukiwarce "web uuid generate"; warto zapamiętać wartość szesnastkową
    private static final UUID SIMPLE_BT_APP_UUID = UUID.fromString("0dfb786a-cafe-feed-cafe-982fdfe4bcbf");
    private static final String SIMPLE_BT_NAME = "SimpleBT";
    // identyfikator okna dialogowego
    private static final int DEVICE_PICKER_DIALOG = 1001;
    // do wymiany danych pomiędzy wątkami
    private final Handler handler = new Handler();
    private BluetoothAdapter btAdapter;
    private BtReceiver btReceiver;
    private ServerListenThread serverListenThread;
    private ClientConnectThread clientConnectThread;
    private BluetoothDataCommThread bluetoothDataCommThread;
    private BluetoothDevice remoteDevice;
    private BluetoothSocket activeBluetoothSocket;
    
    // na potrzeby odtwarzania dźwięków
    private MediaPlayer player;
    private HashMap<String, BluetoothDevice> discoveredDevices = new HashMap<String, BluetoothDevice>();

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.main);
        btAdapter = BluetoothAdapter.getDefaultAdapter();
        if (btAdapter == null) {
            // w urządzeniu nie jest odstępna komunikacja bluetooth
            setStatus("Komunikacja bluetooth nie jest dostępna. :(");
            disableAllButtons();
        } else {
            setStatus("Komunikacja bluetooth dostępna! :)");
            // potrzebujemy odbiorcę
            btReceiver = new BtReceiver();
            // odbieramy zdarzenia zmiany stanu 
            IntentFilter stateChangedFilter = new IntentFilter(BluetoothAdapter.ACTION_STATE_CHANGED);
            this.registerReceiver(btReceiver, stateChangedFilter);
            // odbieramy zdarzenia związane z odkrywaniem innych urządzeń bluetooth
            IntentFilter actionFoundFilter = new IntentFilter(BluetoothDevice.ACTION_FOUND);
            registerReceiver(btReceiver, actionFoundFilter);
            // sprawdzenie stanu
            int currentState = btAdapter.getState();
            setUIForBTState(currentState);
            if (currentState == BluetoothAdapter.STATE_ON) {
                findDevices();
            }
        }
    }

    private void findDevices() {
        String lastUsedRemoteDevice = getLastUsedRemoteBTDevice();
        if (lastUsedRemoteDevice != null) {
            setStatus("Sprawdzam znane sparowane urządzenie, a konkretnie: "+lastUsedRemoteDevice);
            // sprawdzamy czy to urządzenie znajduje się na liście aktualnie widocznych (?), sparowanych urządzeń
            Set<BluetoothDevice> pairedDevices = btAdapter.getBondedDevices();
            for (BluetoothDevice pairedDevice : pairedDevices) {
                if (pairedDevice.getAddress().equals(lastUsedRemoteDevice)) {
                    setStatus("Znaleziono urządzenie: " + pairedDevice.getName() + "@" + lastUsedRemoteDevice);
                    remoteDevice = pairedDevice;
                }
            }
        } 
       
        if (remoteDevice == null) {
            setStatus("Rozpoczynam proces odkrywania...");
            // rozpoczęcie odkrywania
            if (btAdapter.startDiscovery()) {
                setStatus("Odkrywanie rozpoczęte...");
            }


        }
        
        // zezwalamy na odkrywanie tego urządzenia
        setStatus("Zapewnienie możliwości odkrycia tego urządzenia, zostanie wyświetlone okno dialogowe...");
        Intent discoverMe = new Intent(BluetoothAdapter.ACTION_REQUEST_DISCOVERABLE);
        discoverMe.putExtra(BluetoothAdapter.EXTRA_DISCOVERABLE_DURATION, BT_DISCOVERABLE_DURATION);
        startActivity(discoverMe);
        
        // zaczynamy także nasłuchiwać odsyłanych połączeń
        setStatus("Włączenie wątku gniazda nasłuchującego");
        serverListenThread = new ServerListenThread();
        serverListenThread.start();
    }

    private String getLastUsedRemoteBTDevice() {
        SharedPreferences prefs = getPreferences(MODE_PRIVATE);
        String result = prefs.getString("LAST_REMOTE_DEVICE_ADDRESS", null);
        return result;
    }
    
    private void setLastUsedRemoteBTDevice(String name) {
        SharedPreferences prefs = getPreferences(MODE_PRIVATE);
        Editor edit = prefs.edit();
        edit.putString("LAST_REMOTE_DEVICE_ADDRESS", name);
        edit.commit();
    }

    private void disableAllButtons() {
        Button button;
        int[] buttonIds = { R.id.bt_toggle, R.id.connect, R.id.marco, R.id.ping };
        for (int buttonId : buttonIds) {
            button = (Button) findViewById(buttonId);
            button.setEnabled(false);
        }
    }

    private ToggleButton btToggle;

    private void setUIForBTState(int state) {
        if (btToggle == null) {
            btToggle = (ToggleButton) findViewById(R.id.bt_toggle);
        }
        switch (state) {
        case BluetoothAdapter.STATE_ON:
            btToggle.setChecked(true);
            btToggle.setEnabled(true);
            setStatus("Stan BT - włączone");
            break;
        case BluetoothAdapter.STATE_OFF:
            btToggle.setChecked(false);
            btToggle.setEnabled(true);
            setStatus("Stan BT - wyłączone");
            break;
        case BluetoothAdapter.STATE_TURNING_OFF:
            btToggle.setChecked(true);
            btToggle.setEnabled(false);
            setStatus("Stan BT - wyłączanie");
            break;
        case BluetoothAdapter.STATE_TURNING_ON:
            btToggle.setChecked(false);
            btToggle.setEnabled(false);
            setStatus("Stan BT - włączanie");
            break;
        }
    }

    private TextView statusField;

    private void setStatus(String string) {
        if (statusField == null) {
            statusField = (TextView) findViewById(R.id.output_display);
        }
        String current = (String) statusField.getText();
        current = string + "\n" + current;
        // to nie może trwać zbyt długo
        if (current.length() > 1500) {
            int truncPoint = current.lastIndexOf("\n");
            current = (String) current.subSequence(0, truncPoint);
        }
        statusField.setText(current);
    }

    public void doToggleBT(View view) {
        Log.v(DEBUG_TAG, "Wywołano metodę doToggleBT().");
        if (btToggle == null) {
            btToggle = (ToggleButton) findViewById(R.id.bt_toggle);
        }
        // sprawdzenie nowego stanu przycisku, by sprawdzić co chce zrobić użytkownik
        if (btToggle.isChecked() == false) {
            if (serverListenThread != null) {
                serverListenThread.stopListening();
            }
            if (clientConnectThread != null) {
                clientConnectThread.stopConnecting();
            }
            if (bluetoothDataCommThread != null) {
                bluetoothDataCommThread.disconnect();
            }
            btAdapter.cancelDiscovery();
            if (!btAdapter.disable()) {
                setStatus("Nie udało się wyłączyć adaptera.");
            }
            
            remoteDevice = null;
            activeBluetoothSocket = null;
            serverListenThread = null;
            clientConnectThread = null;
            bluetoothDataCommThread = null;
            discoveredDevices.clear();
        } else {
            if (!btAdapter.enable()) {
                setStatus("Nie udało się włączyć adaptera.");
            }
        }
    }

    public void doConnectBT(View view) {
        Log.v(DEBUG_TAG, "Wywołano metodę doConnectBT().");
        if (remoteDevice != null) {
            // nawiązanie połączenia ze zdalnym urządzeniem (remoteDevice)
            doConnectToDevice(remoteDevice);
        } else {
            // pobranie urządzenia, z którym użytkownik chce nawiązać połączenie
            showDialog(DEVICE_PICKER_DIALOG);
        }
    }

    public void doConnectToDevice(BluetoothDevice device) {
        // zatrzymanie procesu odkrywania
        btAdapter.cancelDiscovery();
        setStatus("Uruchomienie wątku nawiązywania połączenia.");
        clientConnectThread = new ClientConnectThread(device);
        clientConnectThread.start();
    }

    public void doStartDataCommThread() {
        if (activeBluetoothSocket == null) {
            setStatus("Nie można uruchomić wątku transmisji danych.");
            Log.w(DEBUG_TAG, "Coś jest źle, nie można próbować korzystać z wątku transmisji danych kiedy nie ma gniazda.");
        } else {
            setStatus("Uruchamiam wątek transmisji danych.");
            bluetoothDataCommThread = new BluetoothDataCommThread(activeBluetoothSocket);
            bluetoothDataCommThread.start();
        }
    }

    public void doSendPing(View view) {
        Log.v(DEBUG_TAG, "Wywołano metodę doSendPing().");
        if (bluetoothDataCommThread != null) {
            bluetoothDataCommThread.send(SIMPLE_BT_COMMAND_PING);
        }
    }

    public void doSendMarco(View view) {
        Log.v(DEBUG_TAG, "Wywołano metodę doSendMarco().");
        if (bluetoothDataCommThread != null) {
            bluetoothDataCommThread.send(SIMPLE_BT_COMMAND_MARCO);
        }
    }
    
    public void doHandleReceivedCommand(String rawCommand) {
        String command = rawCommand.trim();
        setStatus("Otrzymano: "+ command);
        if (command.equals(SIMPLE_BT_COMMAND_PING)) {
            if (player != null) {
                player.release();
            }
            player = new MediaPlayer();
            try {
                player.setDataSource(this, Uri.parse("android.resource://com.androidbook.simplebluetooth/"+R.raw.ping));
                player.prepare();
                player.start();
                bluetoothDataCommThread.send("pong");
            } catch (Exception e) {
                Log.e(DEBUG_TAG, "Nie udało się uruchomić odtwarzania dźwięków.", e);
                setStatus("Odebrano sygnał ping, nie można odtworzyć dźwięków.");
                bluetoothDataCommThread.send("failed");
            }
            
        } else if (command.equals(SIMPLE_BT_COMMAND_MARCO) ) {
            bluetoothDataCommThread.send(SIMPLE_BT_RESPONSE_POLO);
        }
    }

    private class BtReceiver extends BroadcastReceiver {
        @Override
        public void onReceive(Context context, Intent intent) {
            String action = intent.getAction();
            if (action.equals(BluetoothAdapter.ACTION_STATE_CHANGED)) {
                setStatus("Rozgłaszanie: Odebrano ACTION_STATE_CHANGED");
                int currentState = intent.getIntExtra(BluetoothAdapter.EXTRA_STATE, BluetoothAdapter.STATE_OFF);
                setUIForBTState(currentState);
                if (currentState == BluetoothAdapter.STATE_ON) {
                    findDevices();
                }
            } else if (action.equals(BluetoothDevice.ACTION_FOUND)) {
                setStatus("Rozgłaszanie: Odebrano ACTION_FOUND");
                BluetoothDevice foundDevice = intent.getParcelableExtra(BluetoothDevice.EXTRA_DEVICE);
                setStatus("Urządzenie: " + foundDevice.getName() + "@" + foundDevice.getAddress());
                discoveredDevices.put(foundDevice.getName(), foundDevice);
            }
        }
    }

    @Override
    protected void onDestroy() {
        if (serverListenThread != null) {
            serverListenThread.stopListening();
        }
        if (clientConnectThread != null) {
            clientConnectThread.stopConnecting();
        }
        if (bluetoothDataCommThread != null) {
            bluetoothDataCommThread.disconnect();
        }
        if (activeBluetoothSocket != null) {
            try {
                activeBluetoothSocket.close();
            } catch (IOException e) {
                Log.e(DEBUG_TAG, "Nie udało się zamknąć gniazda.", e);
            }
        }
        btAdapter.cancelDiscovery();
        this.unregisterReceiver(btReceiver);
        if (player != null) {
            player.stop();  
            player.release();
            player = null;
        }
        super.onDestroy();
    }

    @Override
    protected Dialog onCreateDialog(int id) {
        Dialog dialog = null;
        switch (id) {
        case DEVICE_PICKER_DIALOG:
            if (discoveredDevices.size() > 0) {
                ListView list = new ListView(this);
                String[] deviceNames = discoveredDevices.keySet()
                        .toArray(new String[discoveredDevices.keySet().size()]);
                ArrayAdapter<String> deviceAdapter = new ArrayAdapter<String>(this,
                        android.R.layout.simple_list_item_1, deviceNames);
                list.setAdapter(deviceAdapter);
                list.setOnItemClickListener(new OnItemClickListener() {
                    @Override
                    public void onItemClick(AdapterView<?> adapter, View view, int position, long id) {
                        String name = (String) ((TextView) view).getText();
                        removeDialog(DEVICE_PICKER_DIALOG);
                        setStatus("Wybrano zdalne urządzenie: " + name);
                        doConnectToDevice(discoveredDevices.get(name));
                    }
                });
                AlertDialog.Builder builder = new AlertDialog.Builder(this);
                builder.setView(list);
                builder.setTitle(R.string.pick_device);
                builder.setNegativeButton(android.R.string.cancel, new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                        removeDialog(DEVICE_PICKER_DIALOG);
                        setStatus("Nie wybrano zdalnego urządzenia.");
                    }
                });
                dialog = builder.create();
            } else {
                setStatus("Nie znaleziono żadnych urządzeń.");
            }
            break;
        }
        return dialog;
    }

    // wątki pomocnicze
    
    // Wątek serwera; nasłuchuje przychodzących połączeń; robią to oba urządzenia biorące udział w komunikacji.
    // Zawsze musi być jeden serwer i jeden klient.
    // Jeśli klient nawiąże połączenie, uzyskujemy obiekt połączenia - BluetoothSocket - którego możemu używać.
    // Jeśli to urządzenie nawiąże połączenie z innym serwerem, przerywamy nasłuchiwanie i użyjemy tego połączenia
    // (będzie to także obiekt BluetoothSocket).
    private class ServerListenThread extends Thread {
        private final BluetoothServerSocket btServerSocket;

        public ServerListenThread() {
            BluetoothServerSocket btServerSocket = null;
            try {
                btServerSocket = btAdapter.listenUsingRfcommWithServiceRecord(SIMPLE_BT_NAME, SIMPLE_BT_APP_UUID);
            } catch (IOException e) {
                Log.e(DEBUG_TAG, "Nie udało się rozpocząć nasłuchu.", e);
            }
            // finalizujemy 
            this.btServerSocket = btServerSocket;
        }

        public void run() {
            BluetoothSocket socket = null;
            try {
                while (true) {
                    handler.post(new Runnable() {
                        public void run() {
                            setStatus("ServerThread: połączenie zaakceptowane.");
                        }
                    });
                    socket = btServerSocket.accept();
                    if (socket != null) {
                        activeBluetoothSocket = socket;
                        // czynności związane z zaakceptowaniem połączenia (wykonywane w osobnym wątku)
                        handler.post(new Runnable() {
                            public void run() {
                                setStatus("Uzyskano gniazdo urządzenia.");
                                doStartDataCommThread();
                            }
                        });
                        btServerSocket.close();
                        break;
                    }
                }
            } catch (Exception e) {
                handler.post(new Runnable() {
                    public void run() {
                        setStatus("Zamknięto gniazdo nasłuchu - błąd lub komunikacja przerwana.");
                    }
                });
            }
        }

        public void stopListening() {
            try {
                btServerSocket.close();
            } catch (Exception e) {
                Log.e(DEBUG_TAG, "Nie udało się zamknąć gniazda nasłuchu.", e);
            }
        }
    }

    // wątek klienta: used to make a synchronous connect call to a device
    private class ClientConnectThread extends Thread {
        private final BluetoothDevice remoteDevice;
        private final BluetoothSocket clientSocket;

        public ClientConnectThread(BluetoothDevice remoteDevice) {
            this.remoteDevice = remoteDevice;
            BluetoothSocket clientSocket = null;
            try {
                clientSocket = remoteDevice.createRfcommSocketToServiceRecord(SIMPLE_BT_APP_UUID);
            } catch (IOException e) {
                Log.e(DEBUG_TAG, "Failed to open local client socket");
            }
            // finalizujemy
            this.clientSocket = clientSocket;
        }

        public void run() {
            boolean success = false;
            try {
                clientSocket.connect();
                success = true;
            } catch (IOException e) {
                Log.e(DEBUG_TAG, "Błąd połączenia klienta lub połączenie anulowano.");
                try {
                    clientSocket.close();
                } catch (IOException e1) {
                    Log.e(DEBUG_TAG, "Nie udało się zamknąc gniazda po błędzie.", e);
                }
            }
            final String status;
            if (success) {
                status = "Nawiązano połączenie ze zdalnym urządzeniem.";
                activeBluetoothSocket = clientSocket;
                // nie musimy już nasłuchiwać
                serverListenThread.stopListening();
            } else {
                status = "Nie udało się nawiązać połączenia ze zdalnym urządzeniem.";
                activeBluetoothSocket = null;
            }
            handler.post(new Runnable() {
                public void run() {
                    setStatus(status);
                    setLastUsedRemoteBTDevice(remoteDevice.getAddress());
                    doStartDataCommThread();
                }
            });
        }

        public void stopConnecting() {
            try {
                clientSocket.close();
            } catch (Exception e) {
                Log.e(DEBUG_TAG, "Nie udało się przerwać nawiązywania połączenia.", e);
            }
        }
    }

    private class BluetoothDataCommThread extends Thread {
        private final BluetoothSocket dataSocket;
        private final OutputStream outData;
        private final InputStream inData;

        public BluetoothDataCommThread(BluetoothSocket dataSocket) {
            this.dataSocket = dataSocket;
            OutputStream outData = null;
            InputStream inData = null;
            try {
                outData = dataSocket.getOutputStream();
                inData = dataSocket.getInputStream();
            } catch (IOException e) {
                Log.e(DEBUG_TAG, "Nie udało się pobrać strumienia wejściowego.", e);
            }
            this.inData = inData;
            this.outData = outData;
        }

        public void run() {
            byte[] readBuffer = new byte[64];
            int readSize = 0;
            try {
                while (true) {
                    readSize = inData.read(readBuffer);

                    final String inStr = new String(readBuffer, 0, readSize);
                    handler.post(new Runnable() {
                       public void run() {
                           doHandleReceivedCommand(inStr);
                       }
                    });
                }
            } catch (Exception e) {
                Log.e(DEBUG_TAG, "Błąd gniazda lub gniazdo zamknięte.",e);
            }
        }

        public boolean send(String out) {
            boolean success = false;
            try {
                outData.write(out.getBytes(), 0, out.length());
                success = true;
            } catch (IOException e) {
                Log.e(DEBUG_TAG, "Nie udało się zapisać i przesłać danych do zdalnego urządzenia.", e);
                setStatus("Błąd przesyłania danych.");
            }
            return success;
        }
        
        public void disconnect() {
            try {
                dataSocket.close();
            } catch (Exception e) {
               Log.e(DEBUG_TAG, "Nie udało się zamknąć gniazda transmisji danych", e);
            }
        }
    }
}